﻿using System;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using Psion.RFID.HF;

namespace HF_Demo
{
    public sealed partial class HyperTerminalForm : Form
    {
        private readonly Reader _reader; //RFID Reader instance

        public HyperTerminalForm(
            Reader reader)
        {
            InitializeComponent();

            if (!Device.IsWM) mainMenu1.Dispose();

            Height = Screen.PrimaryScreen.WorkingArea.Height;
            _reader = reader;
        }

        #region Events

        /// <summary>
        /// Run a command/script
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtRunClick(object sender, EventArgs e)
        {
            // clear
            if (checkBoxClearBeforeSending.Checked) lstVw_Cnsl.Items.Clear();

            //Set stopwatch
            var sw = new Stopwatch();

            try
            {
                string cmd = txtbx_cmd.Text + " " + txtbx_data.Text;

                //Copy command in listview
                lstVw_Cnsl.Items.Add(new ListViewItem("> " + cmd));
                lstVw_Cnsl.Items[lstVw_Cnsl.Items.Count - 1].ForeColor = Color.Gray;

                // Check no empty command
                if (txtbx_cmd.Text.Equals("")) throw new Exception("Empty Cmd not allowed");

                // convert data from HEX string to byte array
                byte[] data = ConvertHelpers.ConvertHexStringToByteArray(txtbx_data.Text);

                //Save the command in the history combo box if not already in the saved list
                bool boo = cbHistory.Items.Cast<string>().Any(savedCmd => savedCmd == cmd);
                if (boo == false) cbHistory.Items.Insert(0, cmd); 
                // remove last element
                if (cbHistory.Items.Count > 10) cbHistory.Items.RemoveAt(cbHistory.Items.Count - 1);
                

                // EXECUTE COMMAND
                sw.Start();
                byte[] buffer = _reader.Protocol(txtbx_cmd.Text, data);
                sw.Stop();

                // check possible error
                RFIDHelpers.CheckErrors(buffer);


                // convert result in Hex-string
                string bufferHexString = ConvertHelpers.ConvertByteArrayToHexString(buffer);

                // show result
                lstVw_Cnsl.Items.Add(new ListViewItem("< (HEX): " + bufferHexString));
                lstVw_Cnsl.Items[lstVw_Cnsl.Items.Count - 1].ForeColor = Color.Black;

                // convert result in ASCII-string
                string bufferString = ConvertHelpers.HexConvertToString(buffer, "");
                lstVw_Cnsl.Items.Add(new ListViewItem(
                                         string.Format("< (ASCII): {0} ", bufferString)));


            }
            catch (InvalidCastException invalidCastException)
            {
                lstVw_Cnsl.Items.Add(new ListViewItem("< " + "Err: " + invalidCastException.Message));
                lstVw_Cnsl.Items.Add(new ListViewItem("Should be hexadecimal value [0..9A..F]"));
                lstVw_Cnsl.Items[lstVw_Cnsl.Items.Count - 1].ForeColor = Color.Red;
            }
            catch (Exception exception)
            {
                lstVw_Cnsl.Items.Add(new ListViewItem("< " + exception.Message));
                lstVw_Cnsl.Items[lstVw_Cnsl.Items.Count - 1].ForeColor = Color.Red;
            }
            finally
            {
                sw.Stop();
                //Stop stopwatch
                lstVw_Cnsl.Items.Add(new ListViewItem(
                                         string.Format("Command Executed in {0:000} ms", sw.ElapsedMilliseconds)));
                lstVw_Cnsl.Items[lstVw_Cnsl.Items.Count - 1].ForeColor = Color.Orange;
            }
        }


        /// <summary>
        /// Handles the SelectedIndexChanged event of the cbHistory control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void CbHistorySelectedIndexChanged(object sender, EventArgs e)
        {
            if (cbHistory.SelectedIndex != -1)
            {
                string[] data = cbHistory.Items[cbHistory.SelectedIndex].ToString().Split(' ');
                txtbx_cmd.Text = data[0];
                txtbx_data.Text = data[1];
            }
        }

        /// <summary>
        /// Clear listview items
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void BtClearClick(object sender, EventArgs e)
        {
            lstVw_Cnsl.Items.Clear();
        }

        /// <summary>
        /// Handles the Click event of the menuItem1 control.
        /// </summary>
        /// <param name="sender">The source of the event.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        private void MenuItem1Click(object sender, EventArgs e)
        {
            Close();
        }

        #endregion
    }
}